/* Copyright 2009 Regents of the University of California
 * All rights reserved.
 */
#include "bilinear.h"
#include <ctype.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#define WOL_FACTOR 1.0
#define MARKER_FACTOR 1.0

#define BUF_SIZE 1024

int strbegins(const char *str, const char *templ)
{
    return strncmp(str, templ, strlen(templ)) == 0;
}

void read_bilinear_coeffs(bilinear_problem_t *b, FILE *fp)
{
    char    buf[BUF_SIZE];
    char    *p;
    int     i, j, l;
    
    fgets(buf, BUF_SIZE, fp);
    if (!strbegins(buf, "BILINEAR")) {
        fprintf(stderr, "No BILINEAR\n");
        exit(EXIT_FAILURE);
    }
    
    b->n = strtol(buf + 8, &p, 10);
    b->m = strtol(p, &p, 10);
    b->k = strtol(p, &p, 10);
    b->functionals = (double *)malloc(b->n * b->m * b->k * sizeof(double));
    b->observeds = (double *)malloc(b->k * sizeof(double));
    b->name = NULL;
    b->functional_names = (char **)malloc(b->k * sizeof(char *));
    
    fgets(buf, BUF_SIZE, fp);
    if (!strbegins(buf, "COEFFICIENTS")) {
        fprintf(stderr, "No COEFFICIENTS\n");
        exit(EXIT_FAILURE);
    }
    
    for (l = 0; l < b->k; ++l) {   
        int len;
        fgets(buf, BUF_SIZE, fp);
        if (!strbegins(buf, "EQUATION")) {
            fprintf(stderr, "No EQUATION\n");
            exit(EXIT_FAILURE);
        }
        
        len = strlen(buf + 9);
        b->functional_names[l] = (char *)malloc(len + 1);
        strcpy(b->functional_names[l], buf + 9);
        b->functional_names[l][len-1] = '\0';
        
        for (i = 0; i < b->n; ++i) {
            fgets(buf, BUF_SIZE, fp);
            p = buf;
            for (j = 0; j < b->m; ++j) {
                b->functionals[(l * b->n + i) * b->m + j] = strtod(p, &p);
            }
        }
    }
}

void read_to_observed(FILE *fp)
{
    char    buf[BUF_SIZE];
    
    do {
        fgets(buf, BUF_SIZE, fp);
    } while(isspace(buf[0]));
    if (!strbegins(buf, "OBSERVED")) {
        fprintf(stderr, "No OBSERVED");
        exit(EXIT_FAILURE);
    }
}

int read_observed_line(bilinear_problem_t *b, FILE *fp)
{
    char    buf[BUF_SIZE];
    char    *p;
    int     i;
    
    if (!fgets(buf, BUF_SIZE, fp)) {
        return FALSE;
    }
    
    for (p = buf; *p && !isspace(*p); ++p)
        ;
    if (b->name != NULL)
        free(b->name);
    b->name = (char *)malloc(p - buf + 1);
    if (!b->name)
        printf("memory allocation failed\n");
    memcpy(b->name, buf, p - buf);
    b->name[p - buf] = '\0';

    if (buf[strlen(buf) - 1] != '\n') {
        fprintf(stderr, "Warning: Buffer not long enough for line %s\n",
                b->name);
    }
    
    for (i = 0; i < b->k; ++i) {
        b->observeds[i] = strtod(p, &p);
    }
    for (i = 13; i < b->k; ++i)
        b->observeds[i] *= MARKER_FACTOR;
    if (30 < b->k)
        b->observeds[30] *= WOL_FACTOR;
    
    return TRUE;
}

void print_bilinear(bilinear_problem_t b, FILE *fp)
{
    int     l, i, j;
    
    fprintf(fp, "BILINEAR %d %d %d\n", b.n, b.m, b.k);
    fprintf(fp, "COEFFICIENTS\n");
    for (l = 0; l < b.k; ++l) {
        fprintf(fp, "EQUATION %d\n", l + 1);
        for (i = 0; i < b.n; ++i) {
            for (j = 0; j < b.m; ++j) {
                fprintf(fp, "%f ", b.functionals[(l * b.n + i) * b.m + j]);
            }
            fprintf(fp, "\n");
        }
    }
    
    fprintf(fp, "\nOBSERVEDS\n%s", b.name);
    for (l = 0; l < b.k; ++l) {
        fprintf(fp, " %f", b.observeds[l]);
    }
    fprintf(fp, "\n");
}
