#!/usr/bin/python
try:
    from numpy import array, identity, transpose
    from numpy.random import rand
    from numpy.linalg import solve, inv
    have_numpy= True
except ImportError:
    have_numpy = False
from math import sqrt
from os import getpid
from random import random
from re import sub
from subprocess import Popen, PIPE
from sys import argv, exit, stderr, stdout

def main():
    types = [("bilinear", make_bilinear, [int, int]),
             ("quadratic", make_quadratic, [int, float]),
             ("unmixed_quadratic", make_unmixed_quadratic, [int]),
             ("rand_bilinear", make_rand_bilinear, [int, int, float]),
             ("prod_linear", make_prod_linear, [int, int, float])]

    filename = "t-%s" % getpid()
    for (name, fn, args) in types:
        if len(argv) < 2 or argv[1] != name:
            continue
        if len(argv) < 2 + len(args):
            usage()
        fn_args = [conv(x) for (conv, x) in zip(args, argv[2:])]
        solns = apply(fn, fn_args + [filename])
        pos_args = argv[2 + len(args):]
        break
    else:
        usage()

    stdout.write("System written to %s\n" % filename)
    run_pos(filename, solns, pos_args, False)

def usage():
    stderr.write("Usage: %s <type> <args> [pos options]\n" % argv[0])
    stderr.write("where <type> <args> is one of:\n")
    stderr.write("    bilinear <n> <m>\n")
    stderr.write("    quadratic <n> <weight>\n")
    stderr.write("    unmixed_quadratic <n>\n")
    stderr.write("    rand_bilinear <n> <m> <weight>\n")
    stderr.write("    prod_linear <n> <m> <weight>\n")
    exit(1)

def run_pos(filename, solns, args, fulldiffs):
    found = [False for i in range(len(solns))]
    p = Popen(["./pos"] + args + [filename], stdout=PIPE)
    out = p.stdout
    line = out.readline()
    result = []
    numvars = len(solns[0])
    while line:
        if line[0:2] == "  ":
            stdout.write(line)
            result.append(float(sub(".*=", "", line.strip())))

            if len(result) is numvars:
                evaluate_result(result, solns, found, fulldiffs)
                result = []
        else:
            stdout.write(line)
            result = []
        line = out.readline()
    p.communicate()
    out.close()
    for i in range(len(solns)):
        if not found[i]:
            stdout.write("missed: %s\n" % solns[i])

def evaluate_result(result, solns, found, fulldiffs):
    for i in range(len(solns)):
        diffs = [r - s for (r, s) in zip(result, solns[i])]
        tol = 0.05 * max(solns[i])
        if max(diffs) < tol and min(diffs) > -tol:
            if fulldiffs:
                stdout.write(str(diffs))
            else:
                dist = sqrt(sum([d * d for d in diffs]))
                stdout.write("%s %s %s" % (min(diffs), max(diffs),dist))
            stdout.write(" %d\n" % i)
            found[i] = True
            break

# Bilinear equations with many solutions

def make_bilinear(n, m, filename):
    if not have_numpy:
        raise ImportError('numpy must be installed to make bilinear equations')

    a = rand(n - 2, m)
    b = rand(n - 2, n - m)

    a = adjust(a)
    b = adjust(b)

    solns = find_solutions(a, b)
    (coeffs, rhs) = bilinear(a, b, True)

    fh = open(filename, "w")
    write_eqns(fh, coeffs, rhs)
    fh.close()
    return solns

def find_solutions(a, b):
    [k, m] = a.shape
    ndx = range(m-1)
    solns = []
    while ndx:
        p = list(intersection([a[i, :] for i in ndx]))
        matrix = []
        for r in range(k):
            if r not in ndx:
                matrix.append(b[r, :])
        p += list(intersection(matrix))
        solns.append(p)
        ndx = incr_ndx(ndx, k)
    return solns

# a is a matrix k x l matrix defining k hyperplanes in R^l. Any l-1 of these
# hyperplanes define a 1-dimensional vector space. We adjust the matrix so that
# each of these 1-dimensional vector spaces are generated by a vector with all
# positive entries.
def adjust(a):
    bounds = find_bounds(a)
    l = a.shape[1]
    shift = array([[-b + 1 for i in range(l)] for b in bounds])
    trans = shift + identity(l)
    # return a * inv(trans)
    return transpose(solve(transpose(trans), transpose(a)))

# returns a vector whose ith entry is the smallest value of the ratio between
# the ith entry of a generating vector of intersection of l-1 rows and the sum
# of all entries
def find_bounds(a):
    l = a.shape[1]
    bounds = [10 for i in range(l)]
    inters = intersections(a)
    for inter in inters:
        for i in range(l):
            if bounds[i] > inter[i]:
                bounds[i] = inter[i]
    return bounds

def intersections(a):
    [k, l] = a.shape
    ndx = range(l-1) # indices of rows
    pts = []
    while ndx:
        matrix = [a[i,:] for i in ndx]
        pts.append(intersection(matrix))
        ndx = incr_ndx(ndx, k)
    return pts

# Given a r x r+1 matrix, it computes a vector in the intersection of the r
# hyperplanes. The sum of the coordinates of the vector will be 1.
def intersection(matrix):
    r = len(matrix)
    matrix.append([1 for i in range(r+1)])
    rhs = array([0 for i in range(r)] + [1])
    return solve(array(matrix), rhs)

# ndx is an array of strictly increasing indices less than max. Increment the
# list to the next such one in lexicographic order
def incr_ndx(ndx, max):
    for i in range(1, len(ndx) + 1):
        if ndx[-i] is not max-i:
            ndx[-i:] = range(ndx[-i] + 1, ndx[-i] + i + 1)
            return ndx
    # at end of the cycle
    return False

def bilinear(a, b, coeffs_nonneg):
    l = a.shape[0]  # number of equations
    coeffs = []
    rhs = []
    min_rhs = 0.05 # minimum value for the right hand side
    for k in range(l):
        min = -min_rhs
        matrix = []
        # it might be possible to do this smarter as a matrix multiplication
        for i in range(a.shape[1]):
            matrix.append([])
            for j in range(b.shape[1]):
                m = a[k, i] * b[k, j]
                matrix[i].append(m)
                if m < min:
                    min = m
        if coeffs_nonneg:
            matrix = [[x - min for x in row] for row in matrix]
        coeffs.append(matrix)
        rhs.append(-min)

    matrix = [[1.0 for i in range(b.shape[1])] for j in range(a.shape[1])]
    coeffs.append(matrix)
    rhs.append(1.0)

    return coeffs, rhs

def write_bilinear(out, coeffs, rhs):
    out.write("BILINEAR %d %d %d\n" % (len(coeffs[0]), len(coeffs[0][0]),
        len(coeffs)))

    out.write("COEFFICIENTS\n")
    ndx = 1
    for matrix in coeffs:
        out.write("EQUATION %d\n" % ndx)
        ndx += 1
        for row in matrix:
            out.write(" ".join([str(coeff) for coeff in row]))
            out.write("\n")
    out.write("\n")

    out.write("OBSERVEDS\n")
    out.write("x")
    for o in rhs:
        out.write(" " + str(o))
    out.write("\n")

def write_eqns(out, coeffs, rhs):
    k = 0
    for matrix in coeffs:
        first = True
        for i in range(len(matrix)):
            if not first:
                out.write("\n")
            for j in range(len(matrix[i])):
                if matrix[i][j] == 0.0:
                    continue

                if first:
                    first = False
                else:
                    out.write(" + ")
                out.write("%s*x%.2d*y%.2d" % (matrix[i][j], i, j))
        out.write(" - %s;\n" % rhs[k])
        k += 1
    out.write(" + ".join(["x%.2d" % i for i in range(len(matrix))]))
    out.write(" - 1.0;\n")

# Random quadratic equations with at least one solution

def make_quadratic(n, weighting, filename):
    x = [random() for i in range(n)]
    fh = open(filename, "w")
    for eqn in range(n):
        sum = 0.0
        terms = []
        for i in range(n):
            coeff = random()
            if i == eqn:
                coeff *= weighting
            terms.append("%s*x%.2d^2" % (coeff, i))
            sum += coeff * x[i] * x[i]

            for j in range(i):
                coeff = random()
                if i == eqn or j == eqn:
                    coeff *= weighting
                terms.append("%s*x%.2d*x%.2d" % (coeff, i, j))
                sum += coeff * x[i] * x[j]
        fh.write("+".join(terms))
        fh.write(" - %s;\n" % sum)

    fh.close()
    return [x]

def make_unmixed_quadratic(n, filename):
    x = [random() for i in range(n)]
    fh = open(filename, "w")
    for eqn in range(n):
        sum = 0.0
        terms = []
        for i in range(n):
            coeff = random()
            terms.append("%s*x%.2d*x%.2d" % (coeff, eqn, i))
            sum += coeff * x[eqn] * x[i]
        fh.write(" + ".join(terms))
        fh.write(" - %s;\n" %sum)
    fh.close()
    return [x]

def make_rand_bilinear(n, m, weight, filename):
    x = [random() for i in range(m)]
    y = [random() for i in range(n-m)]

    fh = open(filename, "w")
    for eqn in range(n):
        eqnsum = 0.0
        terms = []
        for i in range(m):
            for j in range(n-m):
                coeff = random()
                if i == eqn or j == eqn - m:
                    coeff *= weight
                terms.append("%s*x%.2d*y%.2d" % (coeff, i, j))
                eqnsum += coeff * x[i] * y[j]
            coeff = random()
            terms.append("%s*x%.2d" % (coeff, i))
            eqnsum += coeff * x[i]

        fh.write(" + ".join(terms))
        fh.write(" - %s;\n" % eqnsum)
    fh.close()
    return [x + y]

def make_prod_linear(n, m, weight, filename):
    x = [random() for i in range(m)]
    y = [random() for i in range(n-m)]

    sumx = sum(x)
    sumy = sum(y)

    fh = open(filename, "w")
    for eqn in range(n-2):
        a = [random() for i in range(m)]
        b = [random() for i in range(n-m)]
        if eqn < m-1:
            a[eqn] *= weight
        else:
            b[eqn - m + 1] *= weight

        sumax = sum([a[i] * x[i] for i in range(m)])
        sumby = sum([b[j] * y[j] for j in range(n-m)])
        maxa = max(a)
        maxb = max(b)

        terms = []
        for i in range(m):
            for j in range(n-m):
                terms.append("%s*x%.2d*y%.2d" % (a[i] * b[j], i, j))

        for i in range(m):
            coeff = sumby * (maxa - a[i])
            if coeff != 0.0:
                terms.append("%s*x%.2d" % (coeff, i))

        for j in range(n-m):
            coeff = sumax * (maxb - b[j])
            if coeff != 0.0:
                terms.append("%s*y%.2d" % (coeff, j))

        fh.write(" + ".join(terms))
        const = sumby * maxa * sumx + sumax * maxb * sumy - sumax * sumby
        fh.write(" - %s;\n" % const)
    fh.write(" + ".join(["x%.2d" % i for i in range(m)]))
    fh.write(" - %s;\n" % sumx)
    fh.write(" + ".join(["y%.2d" % j for j in range(n-m)]))
    fh.write(" - %s;\n" % sumy)

    return [x + y];

# run main()

if __name__ == "__main__":
    main()
