/* Copyright (C) 2010 Dustin Cartwight
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include "deriv.h"
#include "solve.h"
#include "svd.h"
#include "lu.h"
#include <float.h>
#include <math.h>
#include <stdlib.h>
#include <string.h>

/* Derivative-based methods, i.e. methods which approximate the equations or the
 * KL-divergence with the first few terms of their power series and then apply
 * linear algebra. */

#define MAX_NEWTON_STEPS 100

int certify(problem_t *p, double *x, double *pdelta);
double compute_offset(problem_t *p, double *x);
double jacobian_min_sv(problem_t *p, double *x);
void compute_jacobian(problem_t *p, double *x, gsl_matrix *jacobian);
double max_total_err(problem_t *p, double *x, double delta);
double *monomial_partials(problem_t *p, double *x, double delta, int i, int j);
double sum_monomials(problem_t *p, int eqn, double *monoms);

/* Tests to see if the solution is near an exact solution. Returns:
      TYPE_INEXACT - not near an exact solution
      TYPE_NEAR_EXACT - near exact solution, but Newton's method failed to
                        converge (should rarely occur)
      TYPE_EXACT - has updated x to be an exact solution (roughly up to machine
                   precision) */
int is_exact(problem_t *p, double *x)
{
    double  delta;

    if (!certify(p, x, &delta))
        return TYPE_INEXACT;

    if (newton(p, x, delta))
        return TYPE_EXACT;
    else
        return TYPE_NEAR_EXACT;
}

/* Attempts to certify that there is a real solution near x. If successful, it
 * returns TRUE and sets *pdelta to the radius within which there is guaranteed
 * to be a solution */
int certify(problem_t *p, double *x, double *pdelta)
{
    double  offset = compute_offset(p, x);
    double  min_sv = jacobian_min_sv(p, x);
    double  max_err = 0.0;
    double  delta;

    if (min_sv == 0.0)
        return FALSE;

    do {
        /* Pick a radius slightly larger than what would be needed if the error
         * bound max_err didn't go up */
        delta = 1.1 * (offset + max_err) / min_sv;
        max_err = max_total_err(p, x, delta);

        if (offset < min_sv * delta - max_err) {
            *pdelta = delta;
            return TRUE;
        }
        /* Give up when the quadratic error significantly exceeds the size of
         * the gap */
    } while (max_err < 4.0 * offset);

    return FALSE;
}

/* Computes the (Euclidean) distance by which the equations are non-zero at the
 * specified point */
double compute_offset(problem_t *p, double *x)
{
    double  *monoms = compute_monoms(p, x);
    double  sum_sq = 0.0;
    int     i;

    for (i = 0; i < p->numeqns; i++) {
        double  diff = p->rhs[i] - sum_monomials(p, i, monoms);

        sum_sq += diff * diff;
    }
    
    free(monoms);
    return sqrt(sum_sq);
}

/* Computes the smallest singular value of the Jacobian of the system of
 * equations at the point x */
double jacobian_min_sv(problem_t *p, double *x)
{
    int         i;
    double      min_sv;
    gsl_vector  *sing_vals; /* singular values of Jacobian */
    gsl_matrix  *jacobian;  /* Jacobian of equations */
    gsl_matrix  *scratch;

    /* :TORESOLVE: Should be able to do this when fewer equations than vars */
    if (p->numvars != p->numeqns)
        return 0.0;

    jacobian = gsl_matrix_alloc(p->numvars, p->numvars);
    compute_jacobian(p, x, jacobian);
    sing_vals = gsl_vector_alloc(p->numvars);
    scratch = gsl_matrix_alloc(p->numvars, p->numvars);

    gsl_linalg_SV_decomp_jacobi(jacobian, scratch, sing_vals);

    min_sv = sing_vals->data[0];
    for (i = 1; i < p->numvars; i++) {
        if (sing_vals->data[i] < min_sv)
            min_sv = sing_vals->data[i];
    }

    gsl_matrix_free(jacobian);
    gsl_matrix_free(scratch);
    gsl_vector_free(sing_vals);

    return min_sv;
}

/* Computes the Jacobian of the system of equations. The matrix jacobian should
 * be p->numeqns by p->numvars. */
void compute_jacobian(problem_t *p, double *x, gsl_matrix *jacobian)
{
    int     i, j;

    for (j = 0; j < p->numvars; j++) {
        double  *monom_partials = monomial_partials(p, x, 0.0, j, -1);

        for (i = 0; i < p->numeqns; i++) {
            jacobian->data[i * jacobian->tda + j]
                    = sum_monomials(p, i, monom_partials);
        }

        free(monom_partials);
    }
}

/* Computes the maximum total error for all equations coming from a linear
 * approximation of the functions within a radius delta of x. */
double max_total_err(problem_t *p, double *x, double delta)
{
    double  sum_sq = 0.0;
            /* maximum residual for each function */
    double  *max_residual = (double *)malloc(p->numeqns * sizeof(double));
    int     i, j, eqn;

    for (eqn = 0; eqn < p->numeqns; eqn++)
        max_residual[eqn] = 0.0;

    for (i = 0; i < p->numvars; i++) {
        for (j = 0; j <= i; j++) {
            double  *monom_partials = monomial_partials(p, x, delta, i, j);

            for (eqn = 0; eqn < p->numeqns; eqn++) {
                double  partial = sum_monomials(p, eqn, monom_partials);

                if (i == j)
                    partial /= 2.0;

                if (partial > max_residual[eqn]) 
                    max_residual[eqn] = partial;
            }

            free(monom_partials);
        }
    }

    for (eqn = 0; eqn < p->numeqns; eqn++)
        sum_sq += max_residual[eqn] * max_residual[eqn];

    free(max_residual);
    
    return sqrt(sum_sq) * delta * delta;
}

/* Computes the partial derivatives of the monomials. The vth coordinates is
 * given by x[v] + delta. The derivative is with respect to the ith and jth
 * coordinates, unless j is -1, in which case it is the first partial derivative
 * with respect to i. */
double *monomial_partials(problem_t *p, double *x, double delta, int i, int j)
{
    int     t, v, k;
    double  *partials = malloc(p->numterms * sizeof(double));

    for (t = 0; t < p->numterms; t++) {
        int     ei = p->exps[t * p->numvars + i];
        int     ej = j == -1 ? 1 : p->exps[t * p->numvars + j];

                /* factor coming from exponents */
        partials[t] = (i == j) ? ei * (ei - 1) : ei * ej;
        if (partials[t] == 0.0)
            continue;

        for (v = 0; v < p->numvars; v++) {
            int     e = p->exps[t * p->numvars + v];

            if (v == i) e--;
            if (v == j) e--;
            for (k = 0; k < e; k++)
                partials[t] *= x[v] + delta;
        }
    }

    return partials;
}

double sum_monomials(problem_t *p, int eqn, double *monoms)
{
    int     t;
    double  sum = 0.0;

    for (t = 0; t < p->numterms; t++)
        sum += p->coeffs[eqn * p->numterms + t] * monoms[t];
    return sum;
}

/* Attempts to find an exact solution to problem within radius of x, using
 * Newton's method. If the algorithm converges within radius, then it returns
 * TRUE and otherwise it returns FALSE. If there is a solution within radius, it
 * failure should rarely occur. */
int newton(problem_t *problem, double *start_x, double radius)
{
    gsl_permutation *perm;
    gsl_matrix      *m;
    gsl_vector      *v;
    double          *x;
    int             success = FALSE;
    int             i;

    if (problem->numvars != problem->numeqns)
        return FALSE;

    perm = gsl_permutation_alloc(problem->numvars);
    m = gsl_matrix_alloc(problem->numvars, problem->numvars);
    v = gsl_vector_alloc(problem->numvars);
    x = (double *)malloc(problem->numvars * sizeof(double));

    memcpy(x, start_x, problem->numvars * sizeof(double));

    for (i = 0; i < MAX_NEWTON_STEPS; i++) {
        double  *monoms;
        double  sum_sq = 0.0;
        int     eqn, signum;
        int     converged = TRUE;

        compute_jacobian(problem, x, m);

        monoms = compute_monoms(problem, x);
        for (eqn = 0; eqn < problem->numvars; eqn++) {
            v->data[eqn] = problem->rhs[eqn]
                                - sum_monomials(problem, eqn, monoms);
            if (fabs(v->data[eqn]) > DBL_EPSILON * problem->numvars *
                    problem->rhs[eqn])
                converged = FALSE;
        }
        free(monoms);

        /* Stop when equations have achieved (approximate) equality */
        if (converged) {
            memcpy(start_x, x, problem->numvars * sizeof(double));
            success = TRUE;
            break;
        }

        gsl_linalg_LU_decomp(m, perm, &signum);
        if (gsl_singular(m))
            break;
        gsl_linalg_LU_svx(m, perm, v);

        /* Check that we haven't drifted outside our target area */
        for (eqn = 0; eqn < problem->numvars; eqn++) {
            double  diff = (x[eqn] += v->data[eqn]) - start_x[eqn];

            sum_sq += diff * diff;
        }
        if (sum_sq > radius * radius)
            break;
    }

    gsl_permutation_free(perm);
    gsl_matrix_free(m);
    gsl_vector_free(v);
    free(x);

    return success;
}

/* Estimate distance to local minimum of K-L divergence */
double est_dist_min(problem_t *p, double *x)
{
    double          *monoms = compute_monoms(p, x);
                    /* value of polynomials */
    double          *val = (double *)malloc(p->numeqns * sizeof(double));
                    /* Jacobian of system of equations */
    gsl_matrix      *jac = gsl_matrix_alloc(p->numeqns, p->numvars);
                    /* gradient of the K-L divergence */
    gsl_vector      *grad = gsl_vector_alloc(p->numvars);
                    /* Hessian of the K-L divergence */
    gsl_matrix      *hess = gsl_matrix_alloc(p->numvars, p->numvars);
    gsl_permutation *perm = gsl_permutation_alloc(p->numvars);
    double          sum_sq = 0.0;
    double          max_x = 0.0; /* maximum value in vector x */
    int             signum;
    int             i, j, k;

    for (k = 0; k < p->numeqns; k++)
        val[k] = sum_monomials(p, k, monoms);

    compute_jacobian(p, x, jac);

    /* Compute gradient */
    for (i = 0; i < p->numvars; i++) {
        grad->data[i] = 0.0;

        for (k = 0; k < p->numeqns; k++) {
            double partial = jac->data[k * jac->tda + i];
            grad->data[i] += partial * (1.0 - p->rhs[k] / val[k]);
        }
    }

    /* Compute Hessian */
    for (i = 0; i <  p->numvars; i++) {
        for (j = 0; j <= i; j++) {
            double  h = 0.0;
            double  *monom_partials = monomial_partials(p, x, 0.0, i, j);

            for (k = 0; k < p->numeqns; k++) {
                double  der_eqn = sum_monomials(p, k, monom_partials);
                double  partial_i = jac->data[k * jac->tda + i];
                double  partial_j = jac->data[k * jac->tda + j];
                h += p->rhs[k] * (partial_i * partial_j / val[k] - der_eqn)
                    / val[k] + der_eqn;
            }

            free(monom_partials);

            hess->data[i * hess->tda + j] = h;
            hess->data[j * hess->tda + i] = h;
        }
    }

    /* :TORESOLVE: maybe should check that hess is positive definite? */
    gsl_linalg_LU_decomp(hess, perm, &signum);
    if (gsl_singular(hess)) {
        sum_sq = DBL_MAX;
        max_x = 1.0;
    } else {
        /* Compute local critical point (hopefully minimum) of second order
         * approximation of K-L divergence */
        gsl_linalg_LU_svx(hess, perm, grad);

        for (i = 0; i < p->numvars; i++) {
            sum_sq += grad->data[i] * grad->data[i];
            if (x[i] > max_x)
                max_x = x[i];
        }
    }

    free(monoms);
    free(val);
    gsl_matrix_free(jac);
    gsl_vector_free(grad);
    gsl_matrix_free(hess);
    gsl_permutation_free(perm);

    return sqrt(sum_sq) / max_x;
}
