;;;; Working with vector-polynomials

; Here, a monomial is represented by a vector of integers representing the
; exponents of the variables. A term is a cons of a coefficient and a monomial.
; A polynomial is a list of terms.

(defun multiply-monomials (a b)
  (let* ((num-vars (length a))
         (m (make-array (list num-vars))))
    (dotimes (i num-vars)
      (setf (svref m i) (+ (svref a i) (svref b i))))
    m))

(defun multiply-terms (a b)
  (cons (* (car a) (car b))
        (multiply-monomials (cdr a) (cdr b))))

(defun multiply-onto-fun (term poly acc)
  (do ((acc acc (cons (multiply-terms term (car poly)) acc))
       (poly poly (cdr poly)))
      ((null poly) acc)))

(defmacro multiply-onto (term poly acc)
  `(setf ,acc (multiply-onto-fun ,term ,poly ,acc)))

; Creates a monomial with all 0 exponent vector
(defun all-zero-monomial (num-vars)
  (make-array (list num-vars) :initial-element 0))

(defun num-vars-poly (poly)
  (length (cdar poly))) ; number of variables in first term

;;;; Reducing vector polynomials

; Many of the following functions have a shift parameter, which causes the basis
; to be interpreted on a shifted set of generators

; Does divisor divide monom?
(defmacro is-divisible (monom divisor shift)
  (let ((monom-sym (gensym)))
    `(let ((,monom-sym ,monom))
       (and
         ,@(let ((tests nil))
             (dotimes (i (length divisor) tests)
               (when (> (svref divisor i) 0)
                 (push `(>= (svref ,monom-sym ,(+ i shift))
                           ,(svref divisor i)) tests))))))))

(defun shift-monom (v shift)
  (let ((res (all-zero-monomial (length v))))
    (dotimes (i (- (length v) shift) res)
      (setf (svref res (+ i shift)) (svref v i)))))

(defun shift-term (term shift)
  (cons (car term) (shift-monom (cdr term) shift)))

(defun shift-poly (poly shift)
  (mapcar #'(lambda (term) (shift-term term shift)) poly))

; Takes a Groebner basis and shifts all the variable indies by shift
(defun shift-basis (basis shift)
  (mapcar #'(lambda (poly) (shift-poly poly shift)) basis))

(defun duplicate-basis (dupl basis)
  (mapcan (lambda (shift) (shift-basis basis shift)) dupl))

;;;; Tree polynomials

; A tree polynomial efficiently stores a polynomial such that looking up a
; specific exponent vector is efficient. It is a nested set of lists. The
; position within the ith list gives the exponent of the ith variable. The final
; integer is the coefficient of that term. Connoseurs of enumerative
; combinatorics will notice a parallel between the sequence of cars and cdrs
; necessary to get to a term and the bijective proof that the number of
; monomials of degree d is (n + d - 1 choose d).

; Returns a list of length n, all of whose values are val. It returns a second
; value which is the last cons of the list
(defun list-of-vals (n val)
  (let* ((end (list val))
         (l end))
    (dotimes (i n) (setf l (cons val l)))
    (values l end)))

(defun nth-or-expand (n ptr filler)
  (if (null (car ptr))
      (multiple-value-bind (row res) (list-of-vals n filler)
        (setf (car ptr) row)
        res)
    (do ((n n (1- n))
         (p (car ptr) (cdr p)))
        ((= n 0) p)
      (when (null (cdr p))
        (multiple-value-bind (extn res) (list-of-vals (1- n) filler)
          (setf (cdr p) extn)
          (return res))))))

; Adds a vector term into the tree polynomial tree.
(defun insert-into-tree (coeff monom tree)
  (do ((i 0 (1+ i))
       (ptr tree (nth-or-expand (svref monom i) ptr
                                (if (= i (1- (length monom))) 0 nil))))
      ((>= i (length monom)) (incf (car ptr) coeff))))

(defun insert-poly-into-tree (poly tree)
  (dolist (term poly) (insert-into-tree (car term) (cdr term) tree)))

(defun reduce-insert-into-tree (coeff monom tree reduce-fn)
  (multiple-value-bind (new-coeff new-monom) (funcall reduce-fn coeff monom)
    (unless (= new-coeff 0) 
      (insert-into-tree new-coeff new-monom tree))))

; Creates a tree representing the zero polynomial.
(defun zero-tree () (cons nil nil))

(defun iterate-tree-rec (tree fn monom depth)
  (if (= depth (length monom))
      (funcall fn tree monom)
    (do ((p tree (cdr p))
         (i 0 (1+ i)))
        ((null p))
      (setf (svref monom depth) i)
      (iterate-tree-rec (car p) fn monom (1+ depth)))))

; Iterate over the terms of a tree polynomial, calling fn for each term, with a
; single argument consisting of the term in vector format. Note that the same
; vector is reused throughout the iteration, so it must be copied if it is going
; to be used beyond a single invocation of fn.
(defun iterate-tree (tree num-vars fn)
  (iterate-tree-rec (car tree) fn (all-zero-monomial num-vars) 0))

; Multiplies vector polynomial poly with the tree, reducing with reduce-fn. The
; function reduce-fn should take a coefficient and a monomial and return two
; values of the same type.
(defun multiply-tree (poly tree reduce-fn)
  (let ((result (zero-tree)))
    (iterate-tree tree (num-vars-poly poly)
      (lambda (coeff monom)
        (dolist (term poly)
          (reduce-insert-into-tree (* coeff (car term))
                                   (multiply-monomials monom (cdr term))
                                   result
                                   reduce-fn))))
    result))

(defun copy-vector (v)
  (let* ((l (length v))
         (r (make-array (list l))))
    (dotimes (i l r)
      (setf (svref r i) (svref v i)))))

; Convert a tree into a vector polynomial
(defun tree-to-poly (tree num-vars)
  (let ((poly nil))
    (iterate-tree tree num-vars
      (lambda (coeff monom)
        (unless (= coeff 0)
          (push (cons coeff (copy-vector monom)) poly))))
    poly))

; Convert a vector polynomial into a tree
(defun poly-to-tree (poly)
  (let ((tree (zero-tree)))
    (insert-poly-into-tree poly tree)
    tree))

; Simplify a polynomial by combining like terms
(defun simplify-poly (poly)
  (tree-to-poly (poly-to-tree poly) (num-vars-poly poly)))

; This prunes the tree to remove all remnants of terms with coefficient zero.
; This is destructive on tree.
(defun prune-tree (tree)
  (if (consp tree)
      (let ((left (prune-tree (car tree)))
            (right (prune-tree (cdr tree))))
        (if (and (null right)
                 (or (null left) (eq left 0)))
            nil
          (progn
            (setf (car tree) left)
            (setf (cdr tree) right)
            tree)))
    tree))

; Returns a tree representing the unit in the polynomial ring
(defun one-tree (num-vars)
  (let ((tree (zero-tree)))
    (insert-into-tree 1 (all-zero-monomial num-vars) tree)
    tree))

(defun num-terms (tree)
  (cond
    ((null tree) 0)
    ((consp tree) (+ (num-terms (car tree)) (num-terms (cdr tree))))
    ((= tree 0) 0)
    (t 1)))

;;; Second-generation reduction methods

(defun divide-monomials (numer denom)
  (let* ((num-vars (length numer))
         (result (make-array (list num-vars))))
    (dotimes (i num-vars result)
      (setf (svref result i) (- (svref numer i) (svref denom i))))))

(defun make-fast-reduce-fn (basis)
  (eval
    `(labels
      ((reduction (coeff monom)
        (cond
          ,@(mapcar
              (lambda (poly)
                `((is-divisible monom ,(cdar poly) 0)
                   (values 0 monom)))
              (remove-if (lambda (poly) (cdr poly)) basis))
          ,@(mapcar
              (lambda (poly)
                `((is-divisible monom ,(cdar poly) 0)
                    (reduction
                      (* coeff ,(caadr poly))
                      (multiply-monomials monom
                                          ,(divide-monomials (cdadr poly)
                                                             (cdar poly))))))
              (remove-if (lambda (poly) (or (null (cdr poly)) (cddr poly)))
                         basis))
          (t (values coeff monom)))))
      #'reduction)))

(defun insert-prod-into-tree (poly coeff monom tree fast-reduce-fn)
  (let* ((num-vars (length monom))
         (new-monom (make-array (list num-vars))))
    (dolist (term poly)
      (dotimes (i num-vars)
        (setf (svref new-monom i) (+ (svref monom i) (svref (cdr term) i))))
      (reduce-insert-into-tree (* coeff (car term)) new-monom tree
                               fast-reduce-fn))))

; Divide a polynomial by a monomial
(defun divide-poly (poly divisor)
  (mapcar
    (lambda (term) (cons (car term) (divide-monomials (cdr term) divisor)))
    poly))

(defun make-slow-reduce-fn (basis)
  (let ((slow-basis (remove-if (lambda (poly) (null (cddr poly))) basis)))
    (eval
      `(lambda (coeff monom fast-reduce-fn tree done-tree)
         ,@(unless slow-basis
             '((declare (ignore fast-reduce-fn tree))))
         (block reduce-block
           ,@(mapcar
               (lambda (poly)
                 `(when (is-divisible monom ,(cdar poly) 0)
                    (insert-prod-into-tree (quote ,(divide-poly (cdr poly)
                                                                (cdar poly)))
                                           coeff monom tree fast-reduce-fn)
                    (return-from reduce-block)))
               slow-basis)
           (insert-into-tree coeff monom done-tree))))))

(defun reduce-tree (tree num-vars fast-reduce-fn slow-reduce-fn)
  (do ((new-tree (zero-tree) (zero-tree))
       (tree tree new-tree)
       (done-tree (zero-tree)))
      ((and (null (car tree)) (null (cdr tree))) done-tree)
    (iterate-tree tree num-vars
      (lambda (coeff monom)
        (funcall slow-reduce-fn coeff monom fast-reduce-fn new-tree
                 done-tree)))))

(defun multiply-reduce (poly tree fast-reduce-fn slow-reduce-fn)
  (prune-tree (reduce-tree (multiply-tree poly tree fast-reduce-fn)
                           (num-vars-poly poly)
                           fast-reduce-fn
                           slow-reduce-fn)))

; Computes the product of a list of polynomials, using the given two reduce
; functions to reduce monomials.
(defun product (polys fast-reduce-fn slow-reduce-fn)
  (do ((polys polys (cdr polys))
       (tree (one-tree (num-vars-poly (car polys)))
             (multiply-reduce (car polys) tree fast-reduce-fn slow-reduce-fn)))
      ((null polys) tree)
    (format t "Doing a multiplication~%")))

(defun permute-monom-vars (monom perm)
  (apply #'vector (mapcar #'(lambda (i) (svref monom i)) perm)))

(defun permute-term-vars (term perm)
  (cons (car term) (permute-monom-vars (cdr term) perm)))

(defun permute-poly-vars (poly perm)
  (mapcar #'(lambda (term) (permute-term-vars term perm)) poly))

(defun print-tree (tree vars)
  (let ((num-vars (length vars)))
    (iterate-tree tree num-vars
      #'(lambda (coeff monom)
          (unless (= coeff 0)
            (format t "~A" coeff)
            (dotimes (i num-vars)
              (unless (= (svref monom i) 0)
                (format t " ~A^~A" (nth i vars) (svref monom i))))
            (format t "~%"))))))
