(load "reduction.l")

; monomial = monomial in partitions of the form 1^k

; A partition with at most max-height parts is represented in a tree as a
; exponent vector corresponding to a notional monomial in max-height variables.

; Finds all ways of ading num blocks to the partition part, where all the blocks
; are in index at least min-ndx, the parts have size at most max-width, and no
; two blocks are in the same row. These partitions are added to result, each
; with coefficient coeff.
(defun pieri-rec (result coeff part num min-ndx max-width)
  (if (<= num 0)
    (insert-into-tree coeff part result)
    (do ((i min-ndx (1+ i)))  ; which ndx to put next block
        ((> i (- (length part) num)))
      (when (and (< (aref part i) max-width)
                 (or (= i 0) (< (aref part i) (aref part (1- i)))))
        ; can add a block here
        (incf (aref part i))
        (pieri-rec result coeff part (1- num) (1+ i) max-width)
        (decf (aref part i))))))

; Multiplies a partition in the vector term by the partition 1^height. The
; resulting sum of partitions is added to the tree result.  Partitions with
; entries greater than max-width are dropped from the sum.
(defun pieri (result coeff monom height max-width)
  (pieri-rec result coeff monom height 0 max-width))

; The argument poly is a sum of partitions, stored as a tree. We multiply by the
; partition 1^height and return the resulting sum of paritions as a tree.
(defun pieri-poly (poly height max-height max-width)
  (let ((result (zero-tree)))
    (iterate-tree poly max-height
      #'(lambda (coeff monom)
          (pieri result coeff (copy-vector monom) height max-width)))
    result))

; Converts an exponent vector monom representing a monomial into an expression
; as a sum of partitions.
(defun parts-from-monom (monom max-width)
  (let ((result (one-tree (length monom))))
    (dotimes (i (length monom) result)
      (dotimes (j (aref monom i))
        (setf result
              (pieri-poly result (1+ i) (length monom) max-width))))))

(defun print-partition (part)
  (dotimes (i (length part))
    (dotimes (j (aref part i)) (format t "#"))
    (format t "~%")))

(defun print-sum-partitions (tree max-height)
  (iterate-tree tree max-height
    #'(lambda (coeff monom)
        (unless (= coeff 0)
          (format t "~A~%" coeff)
          (print-partition monom)))))

(defun list-schubert-monoms-rec (acc m i max-width degree)
  (cond
    ((< i 1)
      (if (= degree 0) (cons (copy-vector m) acc) acc))
    (t
      (do ((val 0 (1+ val)))
          ((or (> val max-width) (> (* val i) degree)) acc)
        (setf (aref m (1- i)) val)
        (setf acc (list-schubert-monoms-rec acc m (1- i) (- max-width val)
                                            (- degree (* val i))))))))

; Returns a list of all monomials in at most max-width parts, each no larger
; max-height and which have degree exactly degree.
(defun list-schubert-monoms (max-height max-width degree)
  (list-schubert-monoms-rec nil (all-zero-monomial max-height)
                            max-height max-width degree))

(defun leading-part (monom)
  (let ((part (all-zero-monomial (length monom))))
    (dotimes (i (length monom) part)
      (dotimes (j (1+ i))
        (incf (aref part j) (aref monom i))))))

;;; Working with bases in the Schubert ring

; Returns the coefficient within a tree at given vector index.
(defun coeff-tree (tree index)
  (do ((i 0 (1+ i))
       (tree (car tree) (nth (aref index i) tree)))
      ((>= i (length index)) (or tree 0))))

; Convert a tree to a list of coefficients
(defun flatten-tree (tree basis)
  (mapcar #'(lambda (ndx) (coeff-tree tree ndx)) basis))

(defun transpose (matrix)
  (if matrix
      (apply #'mapcar #'list matrix)
    nil))

; Produces the change of variable matrix to go from the monomial basis to the
; partition basis.
(defun cov-matrix (max-height max-width degree)
  (let* ((monoms (list-schubert-monoms max-height max-width degree))
         (parts (mapcar #'leading-part monoms)))
    (transpose
      (mapcar
        #'(lambda (monom)
            (flatten-tree (parts-from-monom monom max-width) parts))
        monoms))))

; Returns whether or not a given matrix is upper-triangular with 1's on the
; diagonal
(defun std-unipotent? (matrix)
  (do ((i 0 (1+ i))
       (matrix matrix (cdr matrix)))
      ((null matrix) t)
    (do ((j 0 (1+ j))
         (row (car matrix) (cdr row)))
        ((>= j i)
          (unless (= (car row) 1) (return-from std-unipotent? nil)))
      (unless (= (car row) 0) (return-from std-unipotent? nil)))))

; Compute the dot product of two vectors. We assume that w is at least as long
; as v, but it may be longer.
(defun partial-dot (v w)
  (do ((v v (cdr v))
       (w w (cdr w))
       (sum 0 (+ sum (* (car v) (car w)))))
      ((null v) sum)))

; Return a list v such that considered as a column vector, matrix . v is the ith
; (0-based) coordinate vector. In other words, v is the ith column of the
; inverse matrix. We assume that matrix is lower triangular with 1's on the
; diagonal.
(defun col-inv (matrix i)
  (let ((v (list 1)))
    (do ((v-end v (cdr v-end))
         (matrix (nthcdr (1+ i) matrix) (cdr matrix)))
        ((null matrix) (nconc (make-list i :initial-element 0) v))
        (setf (cdr v-end)
              (list (- (partial-dot v (nthcdr i (car matrix)))))))))

; Takes an upper trianguler matrix with 1's on the diagonal as a list of lists
; and returns the inverse of the matrix.
(defun invert-std-unipotent (matrix)
  (let ((trans (transpose matrix)))
    (do ((i (1- (length matrix)) (1- i))
         (inv nil (cons (col-inv trans i) inv)))
        ((< i 0) inv))))

;;; Producing a Groebner basis for a Schubert ring

; Sum of the exponents in a monomial
(defun degree-std (monom)
  (let ((sum 0))
    (dotimes (i (length monom) sum)
      (incf sum (svref monom i)))))

; Product of a matrix and a vector, given as a list of lists and a list,
; respectively.
(defun product-matrix-vector (matrix vect)
  (mapcar #'(lambda (row) (partial-dot row vect)) matrix))

; Produces a list of monomials in 1^k where k is at most max-height, the degree
; is degree and the standard degree is width.
(defun list-boundary-monoms (max-height width degree)
  (delete-if
    #'(lambda (monom) (< (degree-std monom) width))
    (list-schubert-monoms max-height width degree)))

; Takes a list of coefficents and a list of monomials and produces the
; corresponding polynomial.
(defun coeff-vector (coeffs monoms)
  (delete-if
    #'(lambda (term) (= (car term) 0))
    (mapcar #'cons coeffs monoms)))

; Produce the portion of the Groebner basis of the Schubert ring in the degree
; degree.
(defun schubert-groebner-degree (max-height max-width degree)
  (let* ((var (invert-std-unipotent (cov-matrix max-height max-width degree)))
         (monoms (list-schubert-monoms max-height max-width degree))
         (parts (mapcar #'leading-part monoms)))
    (mapcar
      #'(lambda (monom)
          (cons (cons 1 monom)
            (coeff-vector
              (product-matrix-vector var
                (flatten-tree (parts-from-monom monom max-width) parts))
              monoms)))
      (list-boundary-monoms max-height (1+ max-width) degree))))
      

(defun schubert-groebner-basis (max-height max-width)
  (do ((degree (1+ max-width) (1+ degree))
       (basis nil (nconc (schubert-groebner-degree max-height max-width degree)
                         basis)))
      ((> degree (* max-height (1+ max-width))) basis)))

;;; Testing

(defun validate (val descr)
  (unless val (format t "Test failed: ~A~%" descr)))

(validate
  (std-unipotent? (cov-matrix 5 5 5))
  "Change of variables matrix is not unipotent")
(validate
  (equalp (leading-part #(2 1 1 0)) #(4 2 1 0))
  "Leading partition is incorrect")
(validate
  (std-unipotent? (invert-std-unipotent (cov-matrix 5 5 5)))
  "Inverse change of variables matrix is not unipotent")
